import { useState } from "react";
import { Layout } from "@/components/Layout";
import heroImage from "@/assets/hero-hotel.jpg";
import deluxeRoom from "@/assets/room-deluxe.jpg";
import executiveRoom from "@/assets/room-executive.jpg";
import presidentialRoom from "@/assets/room-presidential.jpg";
import restaurantImage from "@/assets/restaurant.jpg";
import poolImage from "@/assets/pool.jpg";
import spaImage from "@/assets/spa.jpg";
import exteriorImage from "@/assets/exterior.jpg";

const galleryImages = [
  { src: heroImage, category: "Lobby", title: "Grand Lobby" },
  { src: deluxeRoom, category: "Rooms", title: "Deluxe Room" },
  { src: executiveRoom, category: "Rooms", title: "Executive Suite" },
  { src: presidentialRoom, category: "Rooms", title: "Presidential Suite" },
  { src: restaurantImage, category: "Dining", title: "Fine Dining Restaurant" },
  { src: poolImage, category: "Amenities", title: "Infinity Pool" },
  { src: spaImage, category: "Amenities", title: "Spa & Wellness" },
  { src: exteriorImage, category: "Exterior", title: "Hotel Exterior" },
];

const categories = ["All", "Lobby", "Rooms", "Dining", "Amenities", "Exterior"];

const Gallery = () => {
  const [activeCategory, setActiveCategory] = useState("All");
  const [selectedImage, setSelectedImage] = useState<string | null>(null);

  const filteredImages = activeCategory === "All" 
    ? galleryImages 
    : galleryImages.filter(img => img.category === activeCategory);

  return (
    <Layout>
      {/* Hero Section */}
      <section className="pt-32 pb-16 bg-primary">
        <div className="container mx-auto px-4 lg:px-8 text-center">
          <span className="text-gold text-sm tracking-[0.2em] uppercase font-medium">
            Visual Journey
          </span>
          <h1 className="text-4xl md:text-5xl font-serif font-bold text-primary-foreground mt-3 mb-4">
            Photo Gallery
          </h1>
          <div className="gold-divider mx-auto" />
          <p className="text-primary-foreground/80 mt-6 max-w-2xl mx-auto">
            Take a virtual tour of our elegant spaces and discover the beauty 
            that awaits you at Hotili.
          </p>
        </div>
      </section>

      {/* Category Filter */}
      <section className="py-8 bg-background border-b border-border">
        <div className="container mx-auto px-4 lg:px-8">
          <div className="flex flex-wrap justify-center gap-3">
            {categories.map((category) => (
              <button
                key={category}
                onClick={() => setActiveCategory(category)}
                className={`px-6 py-2 rounded-full text-sm font-medium transition-all duration-300 ${
                  activeCategory === category
                    ? "bg-gold text-charcoal"
                    : "bg-muted text-muted-foreground hover:bg-gold/20 hover:text-gold"
                }`}
              >
                {category}
              </button>
            ))}
          </div>
        </div>
      </section>

      {/* Gallery Grid */}
      <section className="py-16 bg-background">
        <div className="container mx-auto px-4 lg:px-8">
          <div className="columns-1 md:columns-2 lg:columns-3 gap-6 space-y-6">
            {filteredImages.map((image, index) => (
              <div
                key={index}
                className="group relative break-inside-avoid cursor-pointer overflow-hidden rounded-xl"
                onClick={() => setSelectedImage(image.src)}
              >
                <img
                  src={image.src}
                  alt={image.title}
                  className="w-full object-cover transition-transform duration-700 group-hover:scale-110"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-charcoal/80 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="absolute bottom-0 left-0 right-0 p-6 translate-y-full group-hover:translate-y-0 transition-transform duration-500">
                  <span className="text-gold text-xs tracking-wider uppercase">
                    {image.category}
                  </span>
                  <h3 className="text-cream font-serif text-lg font-semibold mt-1">
                    {image.title}
                  </h3>
                </div>
                {/* Gold overlay on hover */}
                <div className="absolute inset-0 bg-gold/10 opacity-0 group-hover:opacity-100 transition-opacity duration-500 pointer-events-none" />
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Lightbox */}
      {selectedImage && (
        <div
          className="fixed inset-0 z-50 bg-charcoal/95 flex items-center justify-center p-4"
          onClick={() => setSelectedImage(null)}
        >
          <button
            className="absolute top-6 right-6 text-cream hover:text-gold transition-colors text-3xl"
            onClick={() => setSelectedImage(null)}
          >
            ×
          </button>
          <img
            src={selectedImage}
            alt="Gallery"
            className="max-w-full max-h-[90vh] object-contain rounded-lg"
          />
        </div>
      )}
    </Layout>
  );
};

export default Gallery;
