import { Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import deluxeRoom from "@/assets/room-deluxe.jpg";
import executiveRoom from "@/assets/room-executive.jpg";
import presidentialRoom from "@/assets/room-presidential.jpg";
import { useRoomTypes } from "@/hooks/useRoomTypes";

const fallbackRooms = [
  {
    image: deluxeRoom,
    name: "Deluxe Room",
    description: "A perfect blend of comfort and elegance with city views.",
    price: 150,
  },
  {
    image: executiveRoom,
    name: "Executive Suite",
    description: "Spacious suite with separate living area and premium amenities.",
    price: 280,
  },
  {
    image: presidentialRoom,
    name: "Presidential Suite",
    description: "Ultimate luxury with panoramic views and exclusive services.",
    price: 500,
  },
];

export function FeaturedRooms() {
  const { data: roomTypes } = useRoomTypes();

  const rooms = roomTypes?.length
    ? roomTypes.map((roomType, index) => {
        const fallback =
          fallbackRooms[index % fallbackRooms.length]?.image ?? deluxeRoom;

        return {
          image: roomType.image_url || fallback,
          name: roomType.name,
          description:
            roomType.description ||
            "A refined stay with premium comfort and thoughtful amenities.",
          price: Number(roomType.base_price || 0),
          fallbackImage: fallback,
        };
      })
    : fallbackRooms.map((room) => ({ ...room, fallbackImage: room.image }));

  return (
    <section className="py-20 bg-background">
      <div className="container mx-auto px-4 lg:px-8">
        {/* Section Header */}
        <div className="text-center mb-16">
          <span className="text-gold text-sm tracking-[0.2em] uppercase font-medium">
            Accommodations
          </span>
          <h2 className="text-3xl md:text-4xl font-serif font-bold text-foreground mt-3 mb-4">
            Our Elegant Rooms
          </h2>
          <div className="gold-divider" />
          <p className="text-muted-foreground mt-6 max-w-2xl mx-auto">
            Each room is a sanctuary of refined comfort, thoughtfully designed to 
            provide an exceptional stay.
          </p>
        </div>

        {/* Rooms Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {rooms.map((room, index) => (
            <div
              key={index}
              className="group bg-card rounded-xl overflow-hidden shadow-card hover:shadow-elevated transition-all duration-500 gold-border"
            >
              {/* Room Image */}
              <div className="relative h-64 overflow-hidden">
                <img
                  src={room.image}
                  alt={room.name}
                  className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-110"
                  onError={(event) => {
                    event.currentTarget.src = room.fallbackImage;
                  }}
                />
                <div className="absolute inset-0 bg-gradient-to-t from-charcoal/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
              </div>

              {/* Room Info */}
              <div className="p-6">
                <div className="flex items-center justify-between mb-3">
                  <h3 className="text-xl font-serif font-semibold text-foreground">
                    {room.name}
                  </h3>
                  <div className="text-right">
                    <span className="text-gold font-bold text-xl">KSh {room.price.toLocaleString()}</span>
                    <span className="text-muted-foreground text-sm">/night</span>
                  </div>
                </div>
                <p className="text-muted-foreground text-sm mb-6 leading-relaxed">
                  {room.description}
                </p>
                <Button variant="outline" className="w-full">
                  View Details
                </Button>
              </div>
            </div>
          ))}
        </div>

        {/* View All Button */}
        <div className="text-center mt-12">
          <Link to="/rooms">
            <Button variant="gold" size="lg">
              View All Rooms
            </Button>
          </Link>
        </div>
      </div>
    </section>
  );
}
