<?php

namespace App\Services;

use App\Models\Booking;
use App\Models\Room;
use App\Models\Stay;
use Carbon\Carbon;

class RoomAvailabilityService
{
    /**
     * Check room availability for a given room type and date range.
     *
     * @param int $roomTypeId
     * @param Carbon $checkinDate
     * @param Carbon $checkoutDate
     * @return array
     */
    public function checkAvailability(int $roomTypeId, Carbon $checkinDate, Carbon $checkoutDate): array
    {
        // Get total count of rooms for this type (excluding maintenance)
        $totalRooms = Room::where('room_type_id', $roomTypeId)
            ->where('status', '!=', Room::STATUS_MAINTENANCE)
            ->count();

        if ($totalRooms === 0) {
            return [
                'available_rooms_count' => 0,
                'is_available' => false,
            ];
        }

        // Count rooms occupied by overlapping stays
        $occupiedByStays = $this->countRoomsOccupiedByStays($roomTypeId, $checkinDate, $checkoutDate);

        // Count bookings without stays that overlap and need room reservation
        $occupiedByBookings = $this->countRoomsOccupiedByBookings($roomTypeId, $checkinDate, $checkoutDate);

        // Calculate available rooms
        $unavailableRooms = $occupiedByStays + $occupiedByBookings;
        $availableRooms = max(0, $totalRooms - $unavailableRooms);

        return [
            'available_rooms_count' => $availableRooms,
            'is_available' => $availableRooms > 0,
        ];
    }

    /**
     * Count rooms occupied by stays that overlap with the requested date range.
     * 
     * Overlap condition: requested_checkin < existing_checkout AND requested_checkout > existing_checkin
     *
     * @param int $roomTypeId
     * @param Carbon $checkinDate
     * @param Carbon $checkoutDate
     * @return int
     */
    protected function countRoomsOccupiedByStays(int $roomTypeId, Carbon $checkinDate, Carbon $checkoutDate): int
    {
        return Stay::whereHas('room', function ($query) use ($roomTypeId) {
            $query->where('room_type_id', $roomTypeId)
                ->where('status', '!=', Room::STATUS_MAINTENANCE);
        })
            ->where(function ($query) use ($checkinDate, $checkoutDate) {
                // For stays that haven't checked out yet, use the booking's checkout date
                $query->where(function ($q) use ($checkinDate, $checkoutDate) {
                    $q->whereNotNull('checked_out_at')
                        ->where('checked_in_at', '<', $checkoutDate)
                        ->where('checked_out_at', '>', $checkinDate);
                })
                    ->orWhere(function ($q) use ($checkinDate, $checkoutDate) {
                        // Stay is still active (not checked out yet)
                        $q->whereNull('checked_out_at')
                            ->where('checked_in_at', '<', $checkoutDate)
                            ->whereHas('booking', function ($bookingQuery) use ($checkinDate) {
                                $bookingQuery->where('checkout_date', '>', $checkinDate);
                            });
                    });
            })
            ->distinct('room_id')
            ->count('room_id');
    }

    /**
     * Count bookings without stays that overlap with the requested date range.
     * Only count confirmed/pending bookings that haven't been checked in yet.
     *
     * @param int $roomTypeId
     * @param Carbon $checkinDate
     * @param Carbon $checkoutDate
     * @return int
     */
    protected function countRoomsOccupiedByBookings(int $roomTypeId, Carbon $checkinDate, Carbon $checkoutDate): int
    {
        return Booking::where('room_type_id', $roomTypeId)
            ->whereIn('status', [Booking::STATUS_PENDING, Booking::STATUS_CONFIRMED])
            ->doesntHave('stay') // Only bookings without stays (not checked in yet)
            ->where('checkin_date', '<', $checkoutDate)
            ->where('checkout_date', '>', $checkinDate)
            ->count();
    }
}
