<?php

namespace App\Services;

use App\Models\Booking;
use App\Models\Room;
use App\Models\Stay;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class CheckInService
{
    /**
     * Check in a guest by their booking reference.
     *
     * @param string $bookingReference
     * @return Stay|null
     * @throws \Exception
     */
    public function checkInGuest(string $bookingReference): ?Stay
    {
        // Find the booking by reference
        $booking = Booking::where('booking_reference', $bookingReference)
            ->with(['guest', 'roomType'])
            ->first();

        if (!$booking) {
            throw new \Exception("Booking not found with reference: {$bookingReference}");
        }

        // Ensure booking status allows check-in
        if ($booking->status !== Booking::STATUS_CONFIRMED) {
            throw new \Exception("Booking must be confirmed before check-in. Current status: {$booking->status}");
        }

        // Check if booking already has a stay
        if ($booking->stay) {
            throw new \Exception("Booking has already been checked in.");
        }

        // Find an available room of the booking's room type
        $availableRoom = $this->findAvailableRoom(
            $booking->room_type_id,
            $booking->checkin_date,
            $booking->checkout_date,
            $booking->id  // Exclude the current booking from overlap checks
        );

        if (!$availableRoom) {
            throw new \Exception("No rooms available for check-in. Room type: {$booking->roomType->name}");
        }

        // Execute check-in within a transaction
        try {
            return DB::transaction(function () use ($booking, $availableRoom) {
                // Create the Stay record
                $stay = Stay::create([
                    'booking_id' => $booking->id,
                    'room_id' => $availableRoom->id,
                    'checked_in_at' => Carbon::now(),
                    'checked_out_at' => null,
                ]);

                // Update room status to occupied
                $availableRoom->update([
                    'status' => Room::STATUS_OCCUPIED,
                ]);

                // Update booking status to checked_in
                $booking->update([
                    'status' => Booking::STATUS_CHECKED_IN,
                ]);

                return $stay;
            });
        } catch (\Exception $e) {
            throw new \Exception("Check-in failed: " . $e->getMessage());
        }
    }

    /**
     * Find an available room for the given room type and date range.
     * A room is available if:
     * - It belongs to the specified room type
     * - Its status is 'available' (not occupied, cleaning, or maintenance)
     * - It has no overlapping stays or bookings
     *
     * @param int $roomTypeId
     * @param Carbon|string $checkinDate
     * @param Carbon|string $checkoutDate
     * @param int|null $excludeBookingId - Exclude this booking from overlap checks (for current check-in)
     * @return Room|null
     */
    protected function findAvailableRoom(int $roomTypeId, $checkinDate, $checkoutDate, ?int $excludeBookingId = null): ?Room
    {
        $checkinDate = $checkinDate instanceof Carbon ? $checkinDate : Carbon::parse($checkinDate);
        $checkoutDate = $checkoutDate instanceof Carbon ? $checkoutDate : Carbon::parse($checkoutDate);

        // Get all rooms of this type with available status
        $rooms = Room::where('room_type_id', $roomTypeId)
            ->where('status', Room::STATUS_AVAILABLE)
            ->get();

        if ($rooms->isEmpty()) {
            // Debug: Check what statuses exist for this room type
            $allRoomsForType = Room::where('room_type_id', $roomTypeId)->get();
            $statuses = $allRoomsForType->pluck('status')->unique()->join(', ');
            throw new \Exception("No rooms with 'available' status found for this room type. Available statuses: {$statuses}");
        }

        // Find a room without overlapping stays or bookings
        foreach ($rooms as $room) {
            if ($this->isRoomAvailableForDates($room, $checkinDate, $checkoutDate, $excludeBookingId)) {
                return $room;
            }
        }

        return null;
    }

    /**
     * Check if a specific room is available for the given date range.
     * Checks for overlapping stays and confirmed/pending bookings.
     *
     * @param Room $room
     * @param Carbon $checkinDate
     * @param Carbon $checkoutDate
     * @param int|null $excludeBookingId - Exclude this booking from overlap checks
     * @return bool
     */
    protected function isRoomAvailableForDates(Room $room, Carbon $checkinDate, Carbon $checkoutDate, ?int $excludeBookingId = null): bool
    {
        // Check for overlapping stays
        $overlappingStays = Stay::where('room_id', $room->id)
            ->where(function ($query) use ($checkinDate, $checkoutDate) {
                // Stay overlaps if: requested_checkin < existing_checkout AND requested_checkout > existing_checkin
                $query->where(function ($q) use ($checkinDate, $checkoutDate) {
                    // Completed stays
                    $q->whereNotNull('checked_out_at')
                        ->where('checked_in_at', '<', $checkoutDate)
                        ->where('checked_out_at', '>', $checkinDate);
                })
                    ->orWhere(function ($q) use ($checkinDate, $checkoutDate) {
                        // Active stays (not checked out yet)
                        $q->whereNull('checked_out_at')
                            ->where('checked_in_at', '<', $checkoutDate)
                            ->whereHas('booking', function ($bookingQuery) use ($checkinDate) {
                                $bookingQuery->where('checkout_date', '>', $checkinDate);
                            });
                    });
            })
            ->exists();

        if ($overlappingStays) {
            return false;
        }

        // Check for overlapping confirmed/pending bookings without stays
        $overlappingBookingsQuery = Booking::where('room_type_id', $room->room_type_id)
            ->whereIn('status', [Booking::STATUS_PENDING, Booking::STATUS_CONFIRMED])
            ->doesntHave('stay')
            ->where('checkin_date', '<', $checkoutDate)
            ->where('checkout_date', '>', $checkinDate);

        // Exclude the current booking from the overlap check
        if ($excludeBookingId !== null) {
            $overlappingBookingsQuery->where('id', '!=', $excludeBookingId);
        }

        $overlappingBookings = $overlappingBookingsQuery->exists();

        return !$overlappingBookings;
    }
}
