<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\RoomType;
use App\Models\Stay;
use App\Services\CheckInService;
use App\Services\RoomAvailabilityService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class ReceptionDashboardController extends Controller
{
    protected RoomAvailabilityService $availabilityService;
    protected CheckInService $checkInService;

    public function __construct(
        RoomAvailabilityService $availabilityService,
        CheckInService $checkInService
    ) {
        $this->availabilityService = $availabilityService;
        $this->checkInService = $checkInService;
    }

    /**
     * Display the reception dashboard with today's arrivals, departures, and room availability.
     */
    public function index(): Response
    {
        $today = Carbon::today();

        // Get today's arrivals (bookings scheduled to check in today)
        $arrivals = Booking::with(['guest', 'roomType'])
            ->where('checkin_date', $today)
            ->whereIn('status', [Booking::STATUS_CONFIRMED, Booking::STATUS_PENDING])
            ->orderBy('created_at')
            ->get();

        // Get today's departures (bookings scheduled to check out today)
        $departures = Booking::with(['guest', 'roomType', 'stay.room'])
            ->where('checkout_date', $today)
            ->where('status', Booking::STATUS_CHECKED_IN)
            ->orderBy('created_at')
            ->get();

        // Get pending bookings (confirmed but not checked in yet)
        $pendingBookings = Booking::with(['guest', 'roomType'])
            ->whereIn('status', [Booking::STATUS_CONFIRMED, Booking::STATUS_PENDING])
            ->where('checkin_date', '>=', $today)
            ->orderBy('checkin_date')
            ->limit(10)
            ->get();

        // Get room availability by room type
        $roomTypes = RoomType::all();
        $availability = [];

        foreach ($roomTypes as $roomType) {
            $availabilityData = $this->availabilityService->checkAvailability(
                $roomType->id,
                $today,
                $today->copy()->addDay()
            );

            $availability[] = [
                'room_type_id' => $roomType->id,
                'room_type_name' => $roomType->name,
                'available_count' => $availabilityData['available_rooms_count'],
                'is_available' => $availabilityData['is_available'],
            ];
        }

        return Inertia::render('Reception/Dashboard', [
            'arrivals' => $arrivals,
            'departures' => $departures,
            'pendingBookings' => $pendingBookings,
            'availability' => $availability,
            'today' => $today->toDateString(),
        ]);
    }

    /**
     * Display the check-in page with expected check-ins, late check-ins, and search.
     */
    public function checkInIndex(Request $request): Response
    {
        $today = Carbon::today();
        $searchQuery = $request->query('search');
        $period = $request->query('period', 'today'); // today, week, month

        // Calculate date range based on period (week: Sunday to Saturday)
        [$startDate, $endDate] = match($period) {
            'week' => [
                $today->copy()->startOfWeek(Carbon::SUNDAY),
                $today->copy()->endOfWeek(Carbon::SATURDAY)
            ],
            'month' => [
                $today->copy()->startOfMonth(),
                $today->copy()->endOfMonth()
            ],
            default => [$today, $today], // 'today'
        };

        // Expected check-ins (based on period filter, confirmed or pending)
        $expectedCheckIns = Booking::with(['guest', 'roomType'])
            ->whereBetween('checkin_date', [$startDate, $endDate])
            ->whereIn('status', [Booking::STATUS_CONFIRMED, Booking::STATUS_PENDING])
            ->orderBy('checkin_date')
            ->orderBy('created_at')
            ->get()
            ->map(fn (Booking $booking) => [
                'id' => $booking->id,
                'booking_reference' => $booking->booking_reference,
                'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                'guest_phone' => $booking->guest?->phone,
                'room_type_name' => $booking->roomType?->name,
                'checkin_date' => optional($booking->checkin_date)->toDateString(),
                'status' => $booking->status,
            ]);

        // Expected checkouts (based on period filter, currently checked in)
        $expectedCheckouts = Booking::with(['guest', 'roomType', 'stay.room'])
            ->whereBetween('checkout_date', [$startDate, $endDate])
            ->where('status', Booking::STATUS_CHECKED_IN)
            ->orderBy('checkout_date')
            ->orderBy('created_at')
            ->get()
            ->map(fn (Booking $booking) => [
                'id' => $booking->id,
                'booking_reference' => $booking->booking_reference,
                'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                'guest_phone' => $booking->guest?->phone,
                'room_type_name' => $booking->roomType?->name,
                'room_number' => $booking->stay?->room?->room_number,
                'checkout_date' => optional($booking->checkout_date)->toDateString(),
                'status' => $booking->status,
                'stay_id' => $booking->stay?->id,
            ]);

        // Late check-ins (before today, still not checked in)
        $lateCheckIns = Booking::with(['guest', 'roomType'])
            ->where('checkin_date', '<', $today)
            ->whereIn('status', [Booking::STATUS_CONFIRMED, Booking::STATUS_PENDING])
            ->orderBy('checkin_date', 'desc')
            ->get()
            ->map(fn (Booking $booking) => [
                'id' => $booking->id,
                'booking_reference' => $booking->booking_reference,
                'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                'guest_phone' => $booking->guest?->phone,
                'room_type_name' => $booking->roomType?->name,
                'checkin_date' => optional($booking->checkin_date)->toDateString(),
                'status' => $booking->status,
            ]);

        // Search results
        $searchResults = [];
        if ($searchQuery) {
            $searchResults = Booking::with(['guest', 'roomType'])
                ->where(function ($query) use ($searchQuery) {
                    $query->where('booking_reference', 'like', "%{$searchQuery}%")
                        ->orWhereHas('guest', function ($q) use ($searchQuery) {
                            $q->where('first_name', 'like', "%{$searchQuery}%")
                                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                                ->orWhere('phone', 'like', "%{$searchQuery}%");
                        });
                })
                ->whereIn('status', [Booking::STATUS_CONFIRMED, Booking::STATUS_PENDING, Booking::STATUS_CHECKED_IN])
                ->orderBy('checkin_date', 'desc')
                ->limit(20)
                ->get()
                ->map(fn (Booking $booking) => [
                    'id' => $booking->id,
                    'booking_reference' => $booking->booking_reference,
                    'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                    'guest_phone' => $booking->guest?->phone,
                    'room_type_name' => $booking->roomType?->name,
                    'checkin_date' => optional($booking->checkin_date)->toDateString(),
                    'status' => $booking->status,
                ]);
        }

        return Inertia::render('Reception/CheckIn', [
            'expectedCheckIns' => $expectedCheckIns,
            'expectedCheckouts' => $expectedCheckouts,
            'lateCheckIns' => $lateCheckIns,
            'searchResults' => $searchResults,
            'searchQuery' => $searchQuery,
            'period' => $period,
        ]);
    }

    /**
     * Display the checkout page with expected checkouts, overdue checkouts, and search.
     */
    public function checkOutIndex(Request $request): Response
    {
        $today = Carbon::today();
        $searchQuery = $request->query('search');
        $period = $request->query('period', 'today'); // today, week, month

        // Calculate date range based on period (week: Sunday to Saturday)
        [$startDate, $endDate] = match($period) {
            'week' => [
                $today->copy()->startOfWeek(Carbon::SUNDAY),
                $today->copy()->endOfWeek(Carbon::SATURDAY)
            ],
            'month' => [
                $today->copy()->startOfMonth(),
                $today->copy()->endOfMonth()
            ],
            default => [$today, $today], // 'today'
        };

        // Expected checkouts (based on period filter, currently checked in)
        $expectedCheckouts = Booking::with(['guest', 'roomType', 'stay.room'])
            ->whereBetween('checkout_date', [$startDate, $endDate])
            ->where('status', Booking::STATUS_CHECKED_IN)
            ->orderBy('checkout_date')
            ->orderBy('created_at')
            ->get()
            ->map(fn (Booking $booking) => [
                'id' => $booking->id,
                'booking_reference' => $booking->booking_reference,
                'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                'guest_phone' => $booking->guest?->phone,
                'room_type_name' => $booking->roomType?->name,
                'room_number' => $booking->stay?->room?->room_number,
                'checkin_date' => optional($booking->checkin_date)->toDateString(),
                'checkout_date' => optional($booking->checkout_date)->toDateString(),
                'status' => $booking->status,
                'stay_id' => $booking->stay?->id,
            ]);

        // Overdue checkouts (after today, still checked in)
        $overdueCheckouts = Booking::with(['guest', 'roomType', 'stay.room'])
            ->where('checkout_date', '<', $today)
            ->where('status', Booking::STATUS_CHECKED_IN)
            ->orderBy('checkout_date', 'desc')
            ->get()
            ->map(fn (Booking $booking) => [
                'id' => $booking->id,
                'booking_reference' => $booking->booking_reference,
                'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                'guest_phone' => $booking->guest?->phone,
                'room_type_name' => $booking->roomType?->name,
                'room_number' => $booking->stay?->room?->room_number,
                'checkout_date' => optional($booking->checkout_date)->toDateString(),
                'status' => $booking->status,
                'stay_id' => $booking->stay?->id,
            ]);

        // Search results for checkouts
        $searchResults = [];
        if ($searchQuery) {
            $searchResults = Booking::with(['guest', 'roomType', 'stay.room'])
                ->where(function ($query) use ($searchQuery) {
                    $query->where('booking_reference', 'like', "%{$searchQuery}%")
                        ->orWhereHas('guest', function ($q) use ($searchQuery) {
                            $q->where('first_name', 'like', "%{$searchQuery}%")
                                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                                ->orWhere('phone', 'like', "%{$searchQuery}%");
                        });
                })
                ->where('status', Booking::STATUS_CHECKED_IN)
                ->orderBy('checkout_date', 'desc')
                ->limit(20)
                ->get()
                ->map(fn (Booking $booking) => [
                    'id' => $booking->id,
                    'booking_reference' => $booking->booking_reference,
                    'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                    'guest_phone' => $booking->guest?->phone,
                    'room_type_name' => $booking->roomType?->name,
                    'room_number' => $booking->stay?->room?->room_number,
                    'checkout_date' => optional($booking->checkout_date)->toDateString(),
                    'status' => $booking->status,
                    'stay_id' => $booking->stay?->id,
                ]);
        }

        return Inertia::render('Reception/Checkout', [
            'expectedCheckouts' => $expectedCheckouts,
            'overdueCheckouts' => $overdueCheckouts,
            'searchResults' => $searchResults,
            'searchQuery' => $searchQuery,
            'period' => $period,
        ]);
    }

    /**
     * Handle guest check-in via booking reference.
     */
    public function checkIn(Request $request)
    {
        $request->validate([
            'booking_reference' => 'required|string',
        ]);

        try {
            $stay = $this->checkInService->checkInGuest($request->booking_reference);

            return back()->with('success', 'Guest checked in successfully to Room ' . $stay->room->room_number);
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    /**
     * Handle guest check-out.
     */
    public function checkOut(Request $request)
    {
        $request->validate([
            'stay_id' => 'required|exists:stays,id',
        ]);

        try {
            $stay = Stay::with(['booking', 'room'])->findOrFail($request->stay_id);

            // Check if already checked out
            if ($stay->checked_out_at) {
                return back()->with('error', 'Guest has already checked out.');
            }

            // Perform check-out within a transaction
            \DB::transaction(function () use ($stay) {
                // Update stay with checkout timestamp
                $stay->update([
                    'checked_out_at' => Carbon::now(),
                    'actual_checkout_date' => Carbon::today(),
                ]);

                // Update room status to cleaning
                $stay->room->update([
                    'status' => \App\Models\Room::STATUS_CLEANING,
                ]);

                // Update booking status to checked_out
                $stay->booking->update([
                    'status' => Booking::STATUS_CHECKED_OUT,
                ]);
            });

            return back()->with('success', 'Guest checked out successfully from Room ' . $stay->room->room_number);
        } catch (\Exception $e) {
            return back()->with('error', 'Check-out failed: ' . $e->getMessage());
        }
    }

    /**
     * Display the bookings page for receptionist to manage.
     */
    public function bookingsIndex(Request $request): Response
    {
        $search = $request->query('search', '');
        $statusFilter = $request->query('status', 'all');

        $query = Booking::with(['guest', 'roomType']);

        // Apply search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('booking_reference', 'like', "%{$search}%")
                    ->orWhereHas('guest', function ($gq) use ($search) {
                        $gq->where('first_name', 'like', "%{$search}%")
                            ->orWhere('last_name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        // Apply status filter
        if ($statusFilter !== 'all') {
            $query->where('status', $statusFilter);
        }

        $bookings = $query->orderBy('checkin_date', 'desc')->get()->map(fn (Booking $booking) => [
            'id' => $booking->id,
            'reference' => $booking->booking_reference,
            'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
            'guest_phone' => $booking->guest?->phone,
            'guest_email' => $booking->guest?->email,
            'room_type_id' => $booking->room_type_id,
            'room_type_name' => $booking->roomType?->name,
            'checkin_date' => $booking->checkin_date?->toDateString(),
            'checkout_date' => $booking->checkout_date?->toDateString(),
            'status' => $booking->status,
            'number_of_guests' => $booking->number_of_guests,
            'special_requests' => $booking->special_requests,
        ]);

        $statuses = [
            Booking::STATUS_PENDING,
            Booking::STATUS_CONFIRMED,
            Booking::STATUS_CHECKED_IN,
            Booking::STATUS_CHECKED_OUT,
            Booking::STATUS_CANCELLED,
            Booking::STATUS_NO_SHOW,
        ];

        return Inertia::render('Reception/Bookings', [
            'bookings' => $bookings,
            'statuses' => $statuses,
            'search' => $search,
            'statusFilter' => $statusFilter,
        ]);
    }

    /**
     * Update booking status.
     */
    public function updateBookingStatus(Request $request, Booking $booking)
    {
        $request->validate([
            'status' => 'required|string',
        ]);

        $booking->update([
            'status' => $request->status,
        ]);

        return back()->with('success', 'Booking status updated successfully.');
    }
}
