<?php

namespace App\Http\Controllers;

use App\Models\Guest;
use App\Models\RoomType;
use App\Models\Booking;
use App\Services\RoomAvailabilityService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class HomeController extends Controller
{
    protected RoomAvailabilityService $availabilityService;

    public function __construct(RoomAvailabilityService $availabilityService)
    {
        $this->availabilityService = $availabilityService;
    }

    /**
     * Display the public landing page with room types and availability.
     */
    public function index(): Response
    {
        // Get all active room types with details
        $roomTypes = RoomType::orderBy('base_price', 'asc')
            ->get()
            ->map(function (RoomType $roomType) {
                $slugOrId = $roomType->slug ?: $roomType->id;

                return [
                    ...$roomType->toArray(),
                    'image_url' => $roomType->image_path
                        ? Storage::url($roomType->image_path)
                        : url("/images/{$slugOrId}.jpg"),
                ];
            });

        return Inertia::render('Welcome', [
            'roomTypes' => $roomTypes,
            'hotelInfo' => [
                'name' => 'Vienna Crown Resort',
                'tagline' => 'Your Home Away From Home',
                'description' => 'Experience luxury and comfort in the heart of the city. Our modern rooms and exceptional service ensure a memorable stay.',
                'phone' => '+254 700 000 000',
                'email' => 'info@viennacrownresort.com',
                'address' => 'Nairobi, Kenya',
                'socials' => [
                    'instagram' => 'https://instagram.com/viennacrownresort',
                    'facebook' => 'https://facebook.com/viennacrownresort',
                    'x' => 'https://x.com/viennacrownresort',
                    'tiktok' => 'https://tiktok.com/@viennacrownresort',
                ],
            ],
        ]);
    }

    /**
     * Check availability for specific dates and room type.
     */
    public function checkAvailability(Request $request)
    {
        $request->validate([
            'room_type_id' => 'required|exists:room_types,id',
            'checkin_date' => 'required|date|after_or_equal:today',
            'checkout_date' => 'required|date|after:checkin_date',
        ]);

        $checkinDate = Carbon::parse($request->checkin_date);
        $checkoutDate = Carbon::parse($request->checkout_date);

        $availability = $this->availabilityService->checkAvailability(
            $request->room_type_id,
            $checkinDate,
            $checkoutDate
        );

        $roomType = RoomType::findOrFail($request->room_type_id);

        // Calculate number of nights and total price
        $nights = $checkinDate->diffInDays($checkoutDate);
        $totalPrice = $roomType->base_price * $nights;

        return response()->json([
            'available' => $availability['is_available'],
            'available_rooms_count' => $availability['available_rooms_count'],
            'room_type' => $roomType,
            'nights' => $nights,
            'total_price' => $totalPrice,
        ]);
    }

    /**
     * Create a new booking from the public form.
     */
    public function createBooking(Request $request)
    {
        try {
            $validated = $request->validate([
                'room_type_id' => 'required|exists:room_types,id',
                'checkin_date' => 'required|date|after_or_equal:today',
                'checkout_date' => 'required|date|after:checkin_date',
                'number_of_guests' => 'required|integer|min:1',
                'guest_first_name' => 'required|string|max:255',
                'guest_last_name' => 'nullable|string|max:255',
                'guest_phone' => 'required|string|max:255',
                'guest_email' => 'nullable|email|max:255',
                'special_requests' => 'nullable|string',
            ]);

            // Check availability first
            $checkinDate = Carbon::parse($validated['checkin_date']);
            $checkoutDate = Carbon::parse($validated['checkout_date']);

            $availability = $this->availabilityService->checkAvailability(
                $validated['room_type_id'],
                $checkinDate,
                $checkoutDate
            );

            if (!$availability['is_available']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sorry, no rooms available for the selected dates.',
                ], 422);
            }

            // Create or find guest by phone number
            $guest = Guest::firstOrCreate(
                ['phone' => $validated['guest_phone']],
                [
                    'first_name' => $validated['guest_first_name'],
                    'last_name' => $validated['guest_last_name'] ?? null,
                    'email' => $validated['guest_email'],
                ]
            );

            // Calculate total price
            $roomType = RoomType::findOrFail($validated['room_type_id']);
            $nights = $checkinDate->diffInDays($checkoutDate);
            $totalPrice = $roomType->base_price * $nights;

            // Create booking
            $booking = Booking::create([
                'guest_id' => $guest->id,
                'room_type_id' => $validated['room_type_id'],
                'checkin_date' => $checkinDate,
                'checkout_date' => $checkoutDate,
                'number_of_guests' => $validated['number_of_guests'],
                'special_requests' => $validated['special_requests'],
                'total_price' => $totalPrice,
                'status' => Booking::STATUS_PENDING,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Booking created successfully!',
                'booking_reference' => $booking->booking_reference,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Booking creation error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create booking. Please try again.',
            ], 500);
        }
    }
}
