<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use Inertia\Inertia;
use Inertia\Response;

class BookingManagementController extends Controller
{
    public function index(): Response
    {
        $roomTypes = \App\Models\RoomType::select('id', 'name')->get()->toArray();
        return Inertia::render('Admin/Bookings', [
            'bookings' => Booking::query()
                ->with(['guest:id,first_name,last_name,phone,email', 'roomType:id,name'])
                ->latest()
                ->limit(200)
                ->get()
                ->map(fn (Booking $booking) => [
                    'id' => $booking->id,
                    'reference' => $booking->booking_reference,
                    'status' => $booking->status,
                    'guest_name' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                    'guest_phone' => $booking->guest?->phone,
                    'room_type_name' => $booking->roomType?->name,
                    'checkin_date' => optional($booking->checkin_date)->toDateString(),
                    'checkout_date' => optional($booking->checkout_date)->toDateString(),
                    'total_price' => $booking->total_price,
                ]),
            'statuses' => [
                Booking::STATUS_PENDING,
                Booking::STATUS_CONFIRMED,
                Booking::STATUS_CHECKED_IN,
                Booking::STATUS_CHECKED_OUT,
                Booking::STATUS_CANCELLED,
                Booking::STATUS_NO_SHOW,
            ],
            'roomTypes' => $roomTypes,
        ]);
    }

    public function updateStatus(\Illuminate\Http\Request $request, Booking $booking)
    {
        $validated = $request->validate([
            'status' => ['required', 'in:' . implode(',', [
                Booking::STATUS_PENDING,
                Booking::STATUS_CONFIRMED,
                Booking::STATUS_CHECKED_IN,
                Booking::STATUS_CHECKED_OUT,
                Booking::STATUS_CANCELLED,
                Booking::STATUS_NO_SHOW,
            ])],
        ]);

        $booking->update([
            'status' => $validated['status'],
        ]);

        return back()->with('success', 'Booking status updated successfully.');
    }

    public function store(\Illuminate\Http\Request $request)
    {
        $validated = $request->validate([
            'guest_name' => 'required|string|max:255',
            'guest_phone' => 'required|string|max:20',
            'guest_email' => 'nullable|email',
            'room_type_id' => 'required|exists:room_types,id',
            'checkin_date' => 'required|date|after_or_equal:today',
            'checkout_date' => 'required|date|after:checkin_date',
            'number_of_guests' => 'required|integer|min:1',
            'special_requests' => 'nullable|string',
        ]);

        // Create or find guest
        $guest = \App\Models\Guest::firstOrCreate(
            ['phone' => $validated['guest_phone']],
            [
                'first_name' => $validated['guest_name'],
                'email' => $validated['guest_email'] ?? null,
            ]
        );

        // Create booking
        $booking = Booking::create([
            'guest_id' => $guest->id,
            'room_type_id' => $validated['room_type_id'],
            'checkin_date' => $validated['checkin_date'],
            'checkout_date' => $validated['checkout_date'],
            'number_of_guests' => $validated['number_of_guests'],
            'special_requests' => $validated['special_requests'] ?? null,
            'status' => Booking::STATUS_PENDING,
        ]);

        return back()->with('success', 'Booking created successfully.');
    }
}
