<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Room;
use App\Models\RoomType;
use App\Models\User;
use Inertia\Inertia;
use Inertia\Response;

class AdminDashboardController extends Controller
{
    public function index(): Response
    {
        $totalRooms = Room::count();
        $occupiedRooms = Room::where('status', Room::STATUS_OCCUPIED)->count();
        $today = now()->toDateString();

        $billableStatuses = [
            Booking::STATUS_CONFIRMED,
            Booking::STATUS_CHECKED_IN,
            Booking::STATUS_CHECKED_OUT,
        ];

        return Inertia::render('Admin/Dashboard', [
            'stats' => [
                'users' => User::count(),
                'rooms' => Room::count(),
                'roomTypes' => RoomType::count(),
                'availableRooms' => Room::where('status', Room::STATUS_AVAILABLE)->count(),
                'occupiedRooms' => $occupiedRooms,
                'occupancyRate' => $totalRooms > 0 ? round(($occupiedRooms / $totalRooms) * 100, 1) : 0,
                'totalBookings' => Booking::count(),
                'pendingBookings' => Booking::where('status', Booking::STATUS_PENDING)->count(),
                'awaitingCheckin' => Booking::where('status', Booking::STATUS_CONFIRMED)->count(),
                'confirmedBookings' => Booking::where('status', Booking::STATUS_CONFIRMED)->count(),
                'todayCheckins' => Booking::whereDate('checkin_date', $today)->count(),
                'todayCheckouts' => Booking::whereDate('checkout_date', $today)->count(),
                'monthlyRevenue' => (float) Booking::whereBetween('checkin_date', [now()->startOfMonth(), now()->endOfMonth()])
                    ->whereIn('status', $billableStatuses)
                    ->sum('total_price'),
            ],
            'recentBookings' => Booking::with(['guest', 'roomType'])
                ->latest()
                ->limit(8)
                ->get()
                ->map(fn (Booking $booking) => [
                    'id' => $booking->id,
                    'reference' => $booking->booking_reference,
                    'guest' => trim(($booking->guest?->first_name ?? '') . ' ' . ($booking->guest?->last_name ?? '')),
                    'roomType' => $booking->roomType?->name,
                    'status' => $booking->status,
                    'checkinDate' => optional($booking->checkin_date)->toDateString(),
                ]),
        ]);
    }
}
